// Copyright (C) 1998-2000 EPITA-LRDE    
// EPITA Research and Development Laboratory  (lrde@epita.fr) 

#ifndef IMAGE2D_HH
#define IMAGE2D_HH


#include <iostream>
#include "aggregate.hh"
#include "iterator.hh"


template<class Data> class Image2DIterator;
// template<class Data> class Image2DFollowerIterator;
template<class Data> class Image2DNeighborIterator;


const int border = 1;


template<class Data> 
class Image2D : public Aggregate<Data>
{
public:

  Image2D( int nRows, int nCols )
    {
      _nRows = nRows;
      _nCols = nCols;
      _data = new Data*[ _nRows + 2 * border ] + border;
      for ( int iRow = - border; iRow < _nRows + border; ++iRow )
	_data[iRow] = new Data[ _nCols + 2 * border ] + border;
    }

  virtual ~Image2D()
    {
      for ( int iRow = 0; iRow < _nRows; ++iRow )
	delete[] ( _data[iRow] - border );
      delete[] ( _data - border );
    }

  virtual Iterator<Data>& createIterator()
    {
      return *new Image2DIterator<Data>( *this );
    }

  virtual void print() const
    {
      for ( int iRow = 0; iRow < _nRows; ++iRow )
	{
	  for ( int iCol = 0; iCol < _nCols; ++iCol )
	    cout << int( _data[iRow][iCol] ) << ' ';
	  cout << endl;
	}
      cout << endl;	  
    }

  int nRows() const
    {
      return _nRows;
    }

  int nCols() const
    {
      return _nCols;
    }

  unsigned getCard() const
    {
      return _nRows * _nCols;
    }

  Data& value( int iRow, int iCol )
    {
      return _data[iRow][iCol];
    }

  void prettify()
    {
      for ( int iRow = 0; iRow < _nRows; ++iRow )
	for ( int iCol = 0; iCol < _nCols; ++iCol )
	  {
	    _data[iRow][iCol] = iRow * 10 + iCol;
	  }
    }
  
private:

  int     _nRows;
  int     _nCols;
  Data**  _data;
};





struct
{ 
  int dRow;
  int dCol;
}
neighbor2D[4] = { {-1,0}, {0,-1}, {0,1}, {1,0} };  





template<class Data> 
class Image2DNeighborIterator : public Iterator<Data>
{
public:

  Image2DNeighborIterator( Image2DIterator<Data>& iterator ) :
    _iterator( iterator )
    {
      first();
    }

  virtual ~Image2DNeighborIterator()
    {
    }

  virtual void first()
    {
      _v = 0;
    }

  virtual bool isDone() const
    {
      return _v >= 4;
    }

  virtual Data& value()
    {
      return _iterator._image.value( _iterator._iRow + neighbor2D[_v].dRow, 
				     _iterator._iCol + neighbor2D[_v].dCol );
    }

  virtual void next()
    {
      ++_v;
    }

  virtual unsigned getCard() const
    {
      return 4;
    }

  virtual Iterator<Data>& createNeighborIterator()
    {
      return *new Image2DNeighborIterator<Data>( *this );
    }

private:
	
  unsigned _v;
  Image2DIterator<Data>& _iterator;
};





template<class Data> 
class Image2DIterator : public Iterator<Data>
{
public:

  Image2DIterator( Image2D<Data>& image ) :
    _image( image )
    {
      first();
    }

  virtual ~Image2DIterator()
    {
    }

  virtual void first()
    {
      _iRow = _iCol = 0;
    }

  virtual bool isDone() const
    {
      return _iRow >= _image.nRows();
    }

  virtual Data& value()
    {
      return _image.value( _iRow, _iCol );
    }

  virtual void next()
    {
      if ( ++_iCol >= _image.nCols() )  
	{ 
	  _iCol = 0; 
	  ++_iRow;
	}
    }

  virtual unsigned getCard() const
    {
      return _image.getCard();
    }

  virtual Iterator<Data>& createNeighborIterator()
    {
      return *new Image2DNeighborIterator<Data>( *this );
    }

private:
	
  int _iRow;
  int _iCol;
  Image2D<Data>& _image;

  friend class Image2DNeighborIterator<Data>;
};




#endif // IMAGE2D_HH
