// Copyright (C) 1998-2000 EPITA-LRDE    
// EPITA Research and Development Laboratory  (lrde@epita.fr) 

#ifndef IMAGE2D_HH
#define IMAGE2D_HH

#include <iostream>

const int border = 1;


template<class Data> class Image2DIterator;
template<class Data> class Image2DNeighborIterator;


template<class Data> 
class Image2D
{
public:
  typedef Data                           DataType;
  typedef Data*				 Offset;
  typedef signed long			 RelativeOffset;
  typedef Image2DIterator<Data>          Iterator;
  // typedef Image2DFollowerIterator<Data>  FollowerIterator;
  typedef Image2DNeighborIterator<Data>  NeighborIterator;

  Image2D( int nRows, int nCols )
    {
      _nRows = nRows;
      _nCols = nCols;
      _rowSize = _nRows + 2 * border;
      _buffer = new Data[ _rowSize * (_nCols + 2 * border) ];
      _beginOffset = _buffer + _rowSize * border + border;
      _endOffset = _beginOffset + _rowSize * _nCols - 2 * border;
      _data = (new Data*[ _nRows + 2 * border ]) + border;
      for ( int iRow = - border; iRow < _nRows + border; ++iRow )
	_data[iRow] = _beginOffset + iRow * _rowSize;
    }

  ~Image2D()
    {
      delete[] ( _data - border );
      delete[] ( _buffer );
    }

  void print() const
    {
      for ( int iRow = 0; iRow < _nRows; ++iRow )
	{
	  for ( int iCol = 0; iCol < _nCols; ++iCol )
	    cout << int( _data[iRow][iCol] ) << ' ';
	  cout << endl;
	}
      cout << endl;	  
    }

  int nRows() const
    {
      return _nRows;
    }

  int nCols() const
    {
      return _nCols;
    }

  Offset beginOffset() const
    {
      return _beginOffset;
    }

  Offset endOffset() const
    {
      return _endOffset;
    }


  Data& value( int iRow, int iCol )
    {
      return _data[iRow][iCol];
    }

  Data& value( Offset off )
    {
      return *off;
    }

  RelativeOffset relativeOffset( int iRow, int iCol )
    {
      return iRow * _nRows + iCol;
    }

  void prettify()
    {
      for ( int iRow = 0; iRow < _nRows; ++iRow )
	for ( int iCol = 0; iCol < _nCols; ++iCol )
	  {
	    _data[iRow][iCol] = iRow * 10 + iCol;
	  }
    }
  
private:

  int     _nRows;
  int     _nCols;
  unsigned _rowSize;
  Offset  _beginOffset;
  Offset  _endOffset;
  Data**  _data;
  Data*   _buffer;
};





struct
{ 
  int dRow;
  int dCol;
}
neighbor2D[4] = { {-1,0}, {0,-1}, {0,1}, {1,0} };  





template<class Data> 
class Image2DNeighborIterator
{
public:

  enum { card = 4 };

  Image2DNeighborIterator( Image2DIterator<Data>& iterator ) :
    _iterator( iterator )
    {
      for (unsigned i = 0; i < card; ++i)
	neighb[i] = _iterator._image.relativeOffset(neighbor2D[i].dRow,
						    neighbor2D[i].dCol);
      first();
    }

  ~Image2DNeighborIterator()
    {
    }

  void first()
    {
      _v = 0;
    }

  bool isDone() const
    {
      return _v >= card;
    }

  Data& value()
    {
      return _iterator._image.value( _iterator._pos + neighb[_v] );
    }

  void next()
    {
      ++_v;
    }

private:
	
  unsigned _v;
  typename Image2D<Data>::RelativeOffset neighb[card];
  Image2DIterator<Data>& _iterator;
};





template<class Data> 
class Image2DIterator
{
public:

  Image2DIterator( Image2D<Data>& image ) :
    _image( image )
    {
      first();
    }

  ~Image2DIterator()
    {
    }

  void first()
    {
      _pos = _image.beginOffset();
    }

  bool isDone() const
    {
      return _pos >= _image.endOffset();
    }

  Data& value()
    {
      return _image.value( _pos );
    }

  void next()
    {
      ++_pos;
    }

private:
	
  typename Image2D<Data>::Offset _pos;
  Image2D<Data>& _image;

  friend class Image2DNeighborIterator<Data>;
};




#endif // IMAGE2D_HH
