/*
 *  file image2d.hh
 *
 *  Copyright (C) 1999 EPITA-LRDE
 *  EPITA Research and Development Laboratory
 *  lrde@epita.fr
 */

#ifndef IMAGE2D_HH
#define IMAGE2D_HH



template< typename T >  class Image2D_Iterator;



template< typename T > 
class Image2D
{
public:

  typedef  T                    value_type;
  typedef  Image2D_Iterator<T>  iterator_type;

  Image2D( int nrows, int ncols ) :
    _nrows ( nrows ),
    _ncols( ncols )
    {
      _buffer = new T[ size() ];
      _data = new T*[ _nrows ];
      for ( unsigned irow = 0; irow < _nrows; ++irow )
        _data[irow] = _buffer + irow * _ncols;
    }

  ~Image2D()
    {
      delete[] _data;
      delete[] _buffer;
    }

  unsigned nrows() const
    {
      return _nrows;
    }

  unsigned ncols() const
    {
      return _ncols;
    }

  size_t size() const
    {
      return size_t( _nrows ) * size_t( _ncols );
    }
  
  T& value_at( unsigned irow, unsigned icol )
    {
      return _data[irow][icol];
    }
  
  const T& value_at( unsigned irow, unsigned icol ) const
    {
      return _data[irow][icol];
    }

  // misc:

  void prettify()
    {
      for ( unsigned irow = 0; irow < _nrows; ++irow )
        for ( unsigned icol = 0; icol < _ncols; ++icol )
          {
            _data[irow][icol] = irow * 10 + icol;
          }
    }
  
private:

  unsigned  _nrows;
  unsigned  _ncols;
  T**  _data;
  T*   _buffer;
};



template< typename T >
inline ostream& operator<<( ostream& ostr, const Image2D<T>& obj )
{
  for ( unsigned irow = 0; irow < obj.nrows(); ++irow )
    {
      for ( unsigned icol = 0; icol < obj.ncols(); ++icol )
        ostr << obj.value_at( irow, icol ) << ' ';
      ostr << endl;
    }
  return ostr;
}





template< typename T > 
class Image2D_Iterator
{
public:

  Image2D_Iterator( Image2D<T>& image ) :
    _image( image )
    {
      _buffer = &( image.value_at( 0, 0 ) );
      _end_buffer = _buffer + image.size();
      _ptr = 0;
    }

  void first()
    {
      _ptr = _buffer;
    }

  bool isDone() const
    {
      return _ptr == _end_buffer;
    }

  T& operator()()
    {
      return *_ptr;
    }

  const T& operator()() const
    {
      return *_ptr;
    }

  void next()
    {
      ++_ptr;
    }
  
private:
  
  Image2D<T>&  _image;
  T*  _buffer;
  T*  _end_buffer;
  T*  _ptr;

};




#endif
