/*
 * Decompiled with CFR 0.152.
 */
package owl.run;

import com.google.common.base.Throwables;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.ByteChannel;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import owl.run.Environment;
import owl.run.Pipeline;
import owl.run.SimpleExecutionContext;
import owl.run.modules.InputReader;
import owl.run.modules.InputReaders;
import owl.run.modules.OutputWriter;
import owl.run.modules.Transformer;
import owl.run.modules.Transformers;
import owl.util.DaemonThreadFactory;

public final class PipelineRunner {
    private static final Logger logger = Logger.getLogger(PipelineRunner.class.getName());

    private PipelineRunner() {
    }

    public static void run(Pipeline pipeline, Environment env, ByteChannel channel, int worker) throws Exception {
        PipelineRunner.run(pipeline, env, channel, channel, worker);
    }

    public static void run(Pipeline pipeline, Environment env, ReadableByteChannel inputChannel, WritableByteChannel outputChannel, int worker) throws Exception {
        logger.log(Level.FINE, "Instantiating pipeline");
        InputReader reader = pipeline.input();
        List<Transformer.Instance> transformers = Transformers.build(pipeline.transformers(), env);
        logger.log(Level.FINE, "Running pipeline");
        try (ReadableByteChannel readableByteChannel = inputChannel;
             Writer output = Channels.newWriter(outputChannel, StandardCharsets.UTF_8.name());){
            OutputWriter.Binding writer = pipeline.output().bind(output, env);
            if (worker == 0) {
                SequentialRunner.run(env, reader, transformers, writer, inputChannel);
            } else {
                new ParallelRunner(env, reader, transformers, writer, inputChannel, worker).run();
            }
        }
        catch (Exception t) {
            Throwable cause;
            Throwables.throwIfUnchecked((Throwable)t);
            Throwable ex = t;
            while (ex instanceof ExecutionException && (cause = ex.getCause()) != null) {
                ex = cause;
            }
            Throwables.throwIfInstanceOf((Throwable)ex, Exception.class);
            throw new RuntimeException(ex);
        }
        finally {
            env.shutdown();
        }
    }

    @Nullable
    private static Object doTransform(Object input, List<Transformer.Instance> transformers, Supplier<Boolean> earlyStop) throws Exception {
        logger.log(Level.FINEST, "Handling input {0}", input);
        long startTime = System.nanoTime();
        Object output = input;
        for (Transformer.Instance transformer : transformers) {
            SimpleExecutionContext context = new SimpleExecutionContext();
            output = transformer.transform(output, context);
            if (earlyStop.get().booleanValue()) {
                return null;
            }
            String meta = context.getWrittenString();
            if (meta.isEmpty()) continue;
            System.err.print(meta);
        }
        long executionTime = System.nanoTime() - startTime;
        logger.log(Level.FINE, () -> String.format("Execution of transformers for %s took %.2f sec", input, (double)executionTime / (double)TimeUnit.SECONDS.toNanos(1L)));
        return output;
    }

    private static final class ParallelRunner {
        private final AtomicReference<Exception> firstError = new AtomicReference();
        private final AtomicBoolean inputExhausted = new AtomicBoolean(false);
        private final Thread mainThread;
        final ExecutorService executor;
        private final Environment env;
        private final InputReader inputReader;
        final List<Transformer.Instance> transformers;
        private final OutputWriter.Binding outputWriter;
        private final ReadableByteChannel inputChannel;
        private final BlockingQueue<Future<?>> processingQueue = new LinkedBlockingQueue();

        ParallelRunner(Environment env, InputReader inputReader, List<Transformer.Instance> transformers, OutputWriter.Binding writer, ReadableByteChannel inputChannel, int worker) {
            this.env = env;
            this.inputChannel = inputChannel;
            this.mainThread = Thread.currentThread();
            ThreadGroup threadGroup = Thread.currentThread().getThreadGroup();
            int processors = worker > 0 ? worker : Runtime.getRuntime().availableProcessors();
            logger.log(Level.FINER, "Using {0} workers", processors);
            this.executor = Executors.newFixedThreadPool(processors, new DaemonThreadFactory(threadGroup));
            logger.log(Level.FINE, "Instantiating pipeline");
            this.inputReader = inputReader;
            this.transformers = transformers;
            this.outputWriter = writer;
        }

        void run() throws Exception {
            Thread readerThread = new Thread(this.mainThread.getThreadGroup(), this::read, "owl-reader");
            readerThread.setDaemon(true);
            readerThread.setUncaughtExceptionHandler((thread, exception) -> {
                logger.log(Level.SEVERE, "Uncaught exception in reader thread!", exception);
                System.exit(1);
            });
            readerThread.start();
            this.mainThread.setName("owl-writer");
            try {
                this.write();
                logger.log(Level.FINE, "Execution finished");
            }
            finally {
                this.executor.shutdown();
            }
            Exception error = this.firstError.get();
            if (error != null) {
                throw error;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void read() {
            try (Reader reader = Channels.newReader(this.inputChannel, StandardCharsets.UTF_8.name());){
                this.inputReader.run(reader, this.env, input -> this.processingQueue.add(this.executor.submit(new TransformerExecution(input))));
                logger.log(Level.FINE, "Input stream exhausted, waiting for termination");
            }
            catch (Exception t) {
                this.onError(t);
            }
            finally {
                this.inputExhausted.set(true);
                this.mainThread.interrupt();
            }
        }

        private void write() {
            while (!this.inputExhausted.get() || !this.processingQueue.isEmpty()) {
                Object result;
                Future<?> first;
                try {
                    first = this.processingQueue.take();
                }
                catch (InterruptedException ignored) {
                    continue;
                }
                try {
                    result = Uninterruptibles.getUninterruptibly(first);
                }
                catch (ExecutionException e) {
                    this.onError(e);
                    break;
                }
                logger.log(Level.FINEST, "Got result {0} from queue", result);
                if (result == null) {
                    assert (this.hasError());
                    break;
                }
                System.err.flush();
                try {
                    this.outputWriter.write(result);
                }
                catch (Exception e) {
                    this.onError(e);
                    break;
                }
            }
        }

        private void onError(Exception e) {
            logger.log(Level.FINE, "Got error:", e);
            if (!this.firstError.compareAndSet(null, e)) {
                return;
            }
            logger.log(Level.FINER, "Clearing queue after error");
            this.processingQueue.forEach(future -> future.cancel(true));
            this.processingQueue.clear();
            this.inputExhausted.set(true);
            try {
                this.inputChannel.close();
            }
            catch (IOException ex) {
                logger.log(Level.INFO, "IOException after closing input channel", ex);
            }
        }

        boolean hasError() {
            return this.firstError.get() != null;
        }

        private class TransformerExecution
        implements Callable<Object> {
            private final Object input;

            TransformerExecution(Object input) {
                this.input = input;
            }

            @Override
            @Nullable
            public Object call() throws Exception {
                return PipelineRunner.doTransform(this.input, ParallelRunner.this.transformers, ParallelRunner.this::hasError);
            }
        }
    }

    private static final class SequentialRunner {
        private SequentialRunner() {
        }

        static void run(Environment env, InputReader inputReader, List<Transformer.Instance> transformers, OutputWriter.Binding outputWriter, ReadableByteChannel inputChannel) throws Exception {
            Consumer<Object> readerCallback = InputReaders.checkedCallback(input -> outputWriter.write(PipelineRunner.doTransform(input, transformers, () -> Boolean.FALSE)));
            try (Reader reader = Channels.newReader(inputChannel, StandardCharsets.UTF_8.name());){
                inputReader.run(reader, env, readerCallback);
                logger.log(Level.FINE, "Execution finished");
            }
        }
    }
}

